// --- NEW LOGIC FOR SIDE PANEL ---

// This event fires when the extension is installed or updated.
chrome.runtime.onInstalled.addListener(() => {
    // Set the initial state
    chrome.storage.local.set({ isScraping: false });
    
    // Set the default behavior for the side panel: open on toolbar icon click.
    chrome.sidePanel.setPanelBehavior({ openPanelOnActionClick: true });
    console.log("Extension installed/updated. Side panel behavior set.");
});

// This event fires when the browser starts.
chrome.runtime.onStartup.addListener(() => {
    chrome.storage.local.set({ isScraping: false });
    console.log("Browser started. State reset to idle.");
});

// This listener enables/disables the side panel based on the tab's URL.
chrome.tabs.onUpdated.addListener(async (tabId, info, tab) => {
    if (!tab.url) return;
    const url = new URL(tab.url);
    // Enable the side panel on Google Maps pages
    if (url.hostname === 'www.google.com' && url.pathname.startsWith('/maps')) {
        await chrome.sidePanel.setOptions({
            tabId,
            path: 'popup.html',
            enabled: true
        });
    } else {
        // Disable the side panel on all other pages
        await chrome.sidePanel.setOptions({
            tabId,
            enabled: false
        });
    }
});

// --- END OF NEW SIDE PANEL LOGIC ---


let scrapedData = [];
let isScraping = false;
let activeDownloadId = null; 

// The message listener remains the same, but the 'stop' command is slightly simplified.
chrome.runtime.onMessage.addListener(async (request) => {
    if (request.command === 'start') {
        await startScraping();
    } else if (request.command === 'stop') {
        const tabs = await chrome.tabs.query({ url: "https://www.google.com/maps/*" });
        if (tabs.length > 0) {
            try {
                tabs.forEach(tab => chrome.tabs.sendMessage(tab.id, { command: 'stop-scraping' }));
            } catch (e) {
                console.log("Could not send stop message to content script(s), might be inactive.");
            }
        }
        stopScraping('Stopped by user.');
    } else if (request.command === 'data') {
        if (isScraping) {
            scrapedData.push(request.data);
            chrome.runtime.sendMessage({
                command: 'updateStatus',
                text: `Scraped ${scrapedData.length} results...`
            });
        }
    } else if (request.command === 'finished') {
        stopScraping(`Finished. Scraped ${scrapedData.length} results.`);
    }
});

async function startScraping() {
    // This function is IDENTICAL to the previous version
    if (isScraping) {
        console.log("Scraping is already in progress.");
        return;
    }
    let targetTab = null;
    let tabs = await chrome.tabs.query({ active: true, url: "https://www.google.com/maps/*" });
    if (tabs.length > 0) {
        targetTab = tabs[0];
    } else {
        tabs = await chrome.tabs.query({ url: "https://www.google.com/maps/*" });
        if (tabs.length > 0) {
            targetTab = tabs[0];
        }
    }
    if (!targetTab) {
        chrome.runtime.sendMessage({
            command: 'updateStatus',
            text: 'Error: No Google Maps tab found.',
            finished: true
        });
        return;
    }
    isScraping = true;
    scrapedData = [];
    await chrome.storage.local.set({ isScraping: true });
    await chrome.tabs.update(targetTab.id, { active: true });
    await chrome.windows.update(targetTab.windowId, { focused: true });
    try {
        await chrome.scripting.executeScript({
            target: { tabId: targetTab.id },
            files: ['content.js']
        });
        await chrome.tabs.sendMessage(targetTab.id, { command: 'start-scraping' });
    } catch (error) {
        console.error("Failed to inject script or send message:", error);
        stopScraping("Error: Failed to start scraper.");
    }
}

function stopScraping(finalMessage) {
    if (!isScraping && !finalMessage.includes("Error")) return;

    isScraping = false;
    chrome.storage.local.set({ isScraping: false });

    chrome.runtime.sendMessage({
        command: 'updateStatus',
        text: finalMessage,
        finished: true
    });

    if (scrapedData.length > 0) {
        // The side panel does not need to be closed automatically.
        // The download listener will simply handle the download process.
        downloadCSV();
    }
    
    setTimeout(() => {
        scrapedData = [];
    }, 1000);
}

function downloadCSV() {
    // This function is IDENTICAL to the previous version, but its listener will no longer close a window.
    if (scrapedData.length === 0) return;

    const downloadCompleteListener = (delta) => {
        if (delta.id === activeDownloadId && delta.state && (delta.state.current === 'complete' || delta.state.current === 'interrupted')) {
            // The task is done, we just clean up the listener.
            // No need to close the side panel; the user can do that.
            chrome.downloads.onChanged.removeListener(downloadCompleteListener);
            activeDownloadId = null;
        }
    };

    chrome.downloads.onChanged.addListener(downloadCompleteListener);

    const headers = Object.keys(scrapedData[0]);
    let csvContent = headers.join(',') + '\n';

    scrapedData.forEach(item => {
        const row = headers.map(header => {
            let field = item[header] ? item[header].toString() : '';
            field = field.includes(',') || field.includes('"') ? `"${field.replace(/"/g, '""')}"` : field;
            return field;
        }).join(',');
        csvContent += row + '\n';
    });
    
    const url = 'data:text/csv;charset=utf-8,' + encodeURIComponent(csvContent);
    
    chrome.downloads.download({
        url: url,
        filename: 'google_maps_data.csv',
        saveAs: true
    }, (downloadId) => {
        activeDownloadId = downloadId;
    });
}